/********************************************************************

"THIS SOFTWARE IS SUBJECT TO COPYRIGHT PROTECTION AND IS OFFERED ONLY
PURSUANT TO THE 3DFX FXT1 GENERAL PUBLIC LICENSE. A COPY OF THIS
LICENSE MAY BE OBTAINED FROM THE DISTRIBUTOR OR BY CONTACTING 3DFX
INTERACTIVE INC.  

TO THE EXTENT PERMITTED BY APPLICABLE LAW, THERE IS NO WARRANTY FOR
THIS PROGRAM. THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES PROVIDE THIS
PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR
IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE ENTIRE RISK
AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU. SHOULD
THIS PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY
SERVICING, REPAIR OR CORRECTION.

IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, WILL 3DFX INTERACTIVE,
INC., OR ANY OTHER COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY
AND/OR REDISTRIBUTE THIS PROGRAM OR DERIVATIVE WORKS AS PERMITTED
ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL,
INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR
INABILITY TO USE THIS PROGRAM OR DERIVATIVE WORKS (INCLUDING BUT NOT
LIMITED TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES
SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE OF THIS PROGRAM OR
DERIVATIVE WORKS TO OPERATE WITH ANY OTHER PROGRAMS), EVEN IF SUCH
HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
DAMAGES.

USE, DUPLICATION OR DISCLOSURE BY THE GOVERNMENT IS SUBJECT TO
RESTRICTIONS AS SET FORTH IN SUBDIVISION (C)(1)(II) OF THE RIGHTS IN
TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 252.227-7013,
AND/OR IN SIMILAR OR SUCCESSOR CLAUSES IN THE FAR, DOD OR NASA FAR
SUPPLEMENT. UNPUBLISHED RIGHTS RESERVED UNDER THE COPYRIGHT LAWS OF
THE UNITED STATES.  COPYRIGHT 3DFX INTERACTIVE, INC. 1999, ALL RIGHTS
RESERVED"

********************************************************************/


/*
** ISUBPB.C
** Perform image difference calculations
** 
** 9/02/99 rufus@3dfx.com
** Original release cleanup.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "3dfxpb.h"
#include "fximgpb.h"

#include "fxt1gpl.h"

#define ABS(x) ( (x)<0 ? -(x) : (x) )

int diffR, diffG, diffB;
int diffR2, diffG2, diffB2;

/* used to tell how much brighter one image is than another */
FxI32 sumDR, sumDG, sumDB; 

/*----------------------------------------------------------------------
// perform image subtraction: info1 = info1 - info2
//----------------------------------------------------------------------
*/
int subtract(ImgInfo *info1, ImgInfo *info2, int scale, int verbose)
{
    FxU32 x,y, xend,yend;
    FxI32 r,g,b,rgb, dr,dg,db, *data1,*data2;
    FxI32 r2,g2,b2;

    /* compare sizes and use the minimum size */
    xend = info1->any.width;
    if (xend != info2->any.width) {
        fprintf(stderr,"warning: using smaller image's width\n");
        if (xend > info2->any.width)
            xend = info2->any.width;
    }
    yend = info1->any.height;
    if (yend != info2->any.height) {
        fprintf(stderr,"warning: using smaller image's height\n");
        if (yend > info2->any.height)
            yend = info2->any.height;
    }

    data1 = (FxI32 *)info1->any.data;
    data2 = (FxI32 *)info2->any.data;
    for (y=0; y < yend; y++) {
        for (x=0; x < xend; x++) {
            rgb = data1[x];
            r = (rgb >> 16) & 0xFF;
            g = (rgb >> 8) & 0xFF;
            b = rgb & 0xFF;
            rgb = data2[x];
            r2 = (rgb >> 16) & 0xFF;
            g2 = (rgb >> 8) & 0xFF;
            b2 = rgb & 0xFF;

            dr = ABS(r-r2);
            diffR += dr;
            diffR2 += dr*dr;

            dg = ABS(g-g2);
            diffG += dg;
            diffG2 += dg*dg;

            db = ABS(b-b2);
            diffB += db;
            diffB2 += db*db;

            sumDR += r - r2;
            sumDG += g - g2;
            sumDB += b - b2;

            if (verbose > 1)
            if (dr + dg + db) {
                fprintf(stderr,"pixel[%d,%d] 0x%06x != 0x%06x\n",
                        x,y,data1[x],data2[x]);
            }

            r = (r - r2)*scale + 128;
            g = (g - g2)*scale + 128;
            b = (b - b2)*scale + 128;
            if (r<0) r = 0;
            if (g<0) g = 0;
            if (b<0) b = 0;
            if (r>255) r = 255;
            if (g>255) g = 255;
            if (b>255) b = 255;
            rgb = (r<<16) | (g<<8) | b;
            data1[x] = rgb;
        }
        data1 += info1->any.width;
        data2 += info2->any.width;
    }
    return xend * yend;
}

/*----------------------------------------------------------------------
// MAIN PROGRAM
//----------------------------------------------------------------------
*/
void main(int argc, char **argv)
{
    int n;
    int ppm = 0, output = 1, scale = 1, verbose = 0;
    ImgInfo info1, info2;

    while (--argc > 0 && **++argv == '-') {
        char *token;

        for (token = argv[0] + 1; *token; token++) 
        switch (*token) {
            case 'n':
                output = 0;
                break;
            case 's':
                scale = atoi(*++argv);
                argc--;
                break;
            case 'p':
                ppm = 1;
                break;
            case 'v':
                verbose ++;
                break;
            default:
                fprintf (stderr,"illegal option %c\n", *token);
                break;
        }
    }

    if (argc < 1 || argc > 3) {
        fprintf(stderr, FXT1GPL_TXT);
        fprintf(stderr, "usage: isub [-npv] [-s #] inimage1 [inimage2 [outimage]]\n");
        exit(1);
    }

    /* read the input image */
    imgReadFile(argv[0], &info1);
    /* read the input image, if no input file specified, use stdin */
    if (argc > 1)
        imgReadFile(argv[1], &info2);
    else {
        if ( imgReadInfo( stdin, &info2 ) == FXFALSE ) {
                fprintf(stderr,"Error: reading info from stdin, %s\n",imgGetErrorString());
                exit(2);
        }
        if ( imgReadData( stdin, &info2 ) == FXFALSE ) {
                fprintf(stderr, "Error: reading data from stdin, %s\n",imgGetErrorString());
                exit(4);
        }
    }

    n = subtract(&info1,&info2,scale,verbose);
    if (verbose) {
        double rms = sqrt((float)diffR2/n+(float)diffG2/n+(float)diffB2/n);
        double invn = 1.0 / n;
        double invn2 = invn * invn;
        fprintf(stderr,"average bias (R %f,G %f ,B %f) sum %f\n",
                sumDR * invn, sumDG * invn, sumDB * invn,
                (sumDR + sumDG + sumDB) * invn);
        fprintf(stderr,"           sum\t\tmean\tsum sq\t\tmean sq\tstd dev\n");
        fprintf(stderr,"Red diff =%7d\t%.2f\t%8d\t%.2f\t%.2f\n",
                diffR, diffR * invn,
                diffR2, diffR2 * invn,
                sqrt(diffR2 * invn - diffR * diffR * invn2));
        fprintf(stderr,"Grn diff =%7d\t%.2f\t%8d\t%.2f\t%.2f\n",
                diffG, diffG * invn,
                diffG2, diffG2 * invn,
                sqrt(diffG2 * invn - diffG * diffG * invn2));
        fprintf(stderr,"Blu diff =%7d\t%.2f\t%8d\t%.2f\t%.2f\n",
                diffB, diffB * invn,
                diffB2, diffB2 * invn,
                sqrt(diffB2 * invn - diffB * diffB * invn2));
        fprintf(stderr,"RMS = %.2f\tSNR = %.2f\n",
                        rms,10.0*log10(255.0*255.0/rms));
                        
    }

    /* if no output file specified, use stdout */
    if (output) {
        info1.tgaInfo.yOrigin = 0;
        if (argc > 2)
            imgWriteFile(argv[2],&info1, ppm?IMG_P6:IMG_TGA32,info1.any.data);
        else 
            imgWriteImage( stdout, &info1, ppm?IMG_P6:IMG_TGA32,info1.any.data);
    }   

    free(info1.any.data);
    free(info2.any.data);
}